---
title: Controlnet API Usage Guide
createdAt: Thu Jul 18 2024 06:12:37 GMT+0000 (Coordinated Universal Time)
updatedAt: Thu Jul 18 2024 13:40:04 GMT+0000 (Coordinated Universal Time)
---

# Controlnet API Usage Guide

## Introduction

This document will guide developers on how to use the aonweb library to call the Controlnet API, ControlNet is an advanced neural network architecture that enhances image generation models like Stable Diffusion by allowing fine-grained control over the output

## Prerequisites

- Node.js environment
- `aonweb` library installed
- Valid Aonet APPID

## Basic Usage

### 1. Import Required Modules

```js
import { AI, AIOptions } from 'aonweb';
```

### 2. Initialize AI Instance

```js
const ai_options = new AIOptions({
    appId: 'your_app_id_here',
    dev_mode: true
});

const aonweb = new AI(ai_options);
```

### 3. Prepare Input Data Example

```js
const data = {
   input:{
      "seed": 6,
      "image": "https://replicate.delivery/pbxt/IYQCkyANILbqCWObhtFANxUyuVIMsLw7pyky9eFlz17MBG9c/house.png",
      "scale": 9,
      "steps": 20,
      "prompt": "a modernist house in a nice landscape",
      "structure": "seg",
      "low_threshold": 100,
      "high_threshold": 200,
      "image_resolution": 512
    }
};
```

```js
const data = {
   input:{
      "seed": 6,
      "image": "https://replicate.delivery/pbxt/IYQDgL9mAeNkgdFCTCb0qXKDHnL8a84VArZBRGhBYRsqc1vn/1200.jpeg",
      "scale": 9,
      "steps": 20,
      "prompt": "a metallic cyborg bird",
      "structure": "canny",
      "low_threshold": 100,
      "high_threshold": 200,
      "image_resolution": 512
    }
};
```

```js
const data = {
   input:{
      "seed": 20,
      "image": "https://replicate.delivery/pbxt/IYQLHLFDraqCrjDUoiwpM9xBhQM1eQVHbxBiNxcbwctUamzb/user_1.png",
      "scale": 9,
      "steps": 20,
      "prompt": "a photo of a brightly colored turtle",
      "structure": "scribble",
      "low_threshold": 100,
      "high_threshold": 200,
      "image_resolution": 512
    }
};
```

### 4. Call the AI Model

```js
const price = 8; // Cost of the AI call
try {
    const response = await aonweb.prediction("/predictions/ai/controlnet@jagilley", data, price);
    // Handle response
    console.log("Controlnet result:", response);
} catch (error) {
    // Error handling
    console.error("Error generating :", error);
}
```

### Parameter Description

- `seed` Number,Provide the seeds required for model inference
- `image` String,Please provide the image file that needs to be processed.
- `scale` Number,Scale for classifier-free guidance
- `steps` Number,Provide the steps required for model inference
- `prompt` String, Please provide the prompt that needs to be inferred.
- `structure` String,Controlnet structure to condition on
- `low_threshold` Number,[canny only] Line detection low threshold
- `high_threshold` Number,[canny only] Line detection high threshold
- `image_resolution` String,Resolution of output image (will be scaled to this as its smaller dimension)
- `num_outputs` Number,Number of images to output (higher values may OOM),default:1
- `eta` Number,Controls the amount of noise that is added to the input data during the denoising diffusion process. Higher value -> more noise
- `negative_prompt` String,Provide the negative prompt required for model inference

### Notes

- Ensure that the provided image URL is publicly accessible and of good quality to achieve the best recognition results.
- The API may take some time to process the input and generate the result, consider implementing appropriate wait or loading states.
- Handle possible errors, such as network issues, invalid input, or API limitations.
- Adhere to the terms of use and privacy regulations, especially when handling image samples of others.

### Example Response

The API response will contain the results of the image recognition or other relevant information. Parse and use the response data according to the actual API documentation.


